let mainChart;
let currentChartIndex = 0;
let currentYearIndex = 0;
let isLabMode = false;
let years = [];

const chartConfigs = [
    {
        type: 'bar',
        title: 'Taux de parole par sport',
        labels: ['Wrestling', 'Softball', 'Taekwondo', 'Basketball', 'Baseball', 'Volleyball', 'Swimming', 'Rowing', 'Athletics'],
        datasets: [
            {
                label: 'M',
                backgroundColor: '#4bb2d9',
                barPercentage: 0.6,
                hoverBackgroundColor: '#3a8fb8',
                borderWidth: 0,
                borderRadius: 6,
                hoverBorderWidth: 2,
                hoverBorderColor: '#ffffff'
            },
            {
                label: 'F',
                backgroundColor: '#5d64e6',
                barPercentage: 0.6,
                hoverBackgroundColor: '#4a50c4',
                borderWidth: 0,
                borderRadius: 6,
                hoverBorderWidth: 2,
                hoverBorderColor: '#ffffff'
            },
            {
                label: 'Sans valeur',
                backgroundColor: '#a370f7',
                barPercentage: 0.6,
                hoverBackgroundColor: '#8a5cd4',
                borderWidth: 0,
                borderRadius: 6,
                hoverBorderWidth: 2,
                hoverBorderColor: '#ffffff'
            }
        ]
    },
    {
        type: 'line',
        title: 'Évolution mensuelle',
        labels: ['Jan', 'Fév', 'Mar', 'Avr', 'Mai', 'Jun', 'Jul', 'Aoû', 'Sep'],
        datasets: [
            {
                label: 'M',
                backgroundColor: 'rgba(75, 178, 217, 0.1)',
                borderColor: '#4bb2d9',
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointBackgroundColor: '#4bb2d9',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2,
                pointRadius: 6,
                pointHoverRadius: 10
            },
            {
                label: 'F',
                backgroundColor: 'rgba(91, 100, 230, 0.1)',
                borderColor: '#5d64e6',
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointBackgroundColor: '#5d64e6',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2,
                pointRadius: 6,
                pointHoverRadius: 10
            }
        ]
    },
    {
        type: 'radar',
        title: 'Analyse par chaîne',
        labels: ['TF1', 'France 2', 'France 3', 'Canal+', 'Arte', 'M6', 'W9', 'TMC', 'NRJ'],
        datasets: [
            {
                label: 'Couverture homme',
                backgroundColor: 'rgba(75, 178, 217, 0.2)',
                borderColor: '#4bb2d9',
                borderWidth: 2,
                pointBackgroundColor: '#4bb2d9',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2,
                pointRadius: 4
            },
            {
                label: 'Couverture femme',
                backgroundColor: 'rgba(91, 100, 230, 0.2)',
                borderColor: '#5d64e6',
                borderWidth: 2,
                pointBackgroundColor: '#5d64e6',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2,
                pointRadius: 4
            }
        ]
    }
];

document.addEventListener('DOMContentLoaded', function() {
    // Initialiser l'animation TV
    initTVAnimation();
    
    // Initialiser les données
    initData();
    
    // Initialiser les graphiques
    initCharts();
    
    // Configurer les écouteurs d'événements
    setupEventListeners();
    
    // Charger les données initiales
    loadData();
});

function initTVAnimation() {
    const tvAnimation = document.getElementById('tvAnimation');
    const dashboard = document.getElementById('dashboard');
    
    // Simuler l'allumage TV
    setTimeout(() => {
        tvAnimation.style.animation = 'tvPowerOff 0.8s ease-in-out forwards';
        setTimeout(() => {
            tvAnimation.style.display = 'none';
            dashboard.style.opacity = '1';
        }, 800);
    }, 1500);
}

function initData() {
    // Extraire les années disponibles
    years = Array.from(document.querySelectorAll('#yearFilter option')).map(option => option.value);
    currentYearIndex = years.indexOf('2019');
}

function initCharts() {
    const ctx = document.getElementById('mainChart').getContext('2d');
    
    mainChart = new Chart(ctx, {
        type: chartConfigs[currentChartIndex].type,
        data: {
            labels: chartConfigs[currentChartIndex].labels,
            datasets: chartConfigs[currentChartIndex].datasets.map(dataset => ({
                ...dataset,
                data: []
            }))
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: { 
                    display: false 
                },
                tooltip: {
                    enabled: false,
                    external: externalTooltip
                }
            },
            scales: {
                x: {
                    stacked: chartConfigs[currentChartIndex].type === 'bar',
                    grid: { 
                        display: false,
                        color: 'rgba(0, 0, 0, 0.05)'
                    },
                    ticks: { 
                        font: { 
                            size: 11,
                            weight: '500'
                        },
                        color: function(context) {
                            return isLabMode ? '#ffffff' : '#202124';
                        }
                    }
                },
                y: {
                    stacked: chartConfigs[currentChartIndex].type === 'bar',
                    beginAtZero: true,
                    grid: { 
                        color: function(context) {
                            return isLabMode ? 'rgba(255, 255, 255, 0.1)' : 'rgba(0, 0, 0, 0.05)';
                        }
                    },
                    ticks: { 
                        font: { 
                            size: 11,
                            weight: '500'
                        },
                        color: function(context) {
                            return isLabMode ? '#ffffff' : '#202124';
                        }
                    }
                }
            },
            interaction: {
                intersect: false,
                mode: 'index'
            },
            animation: {
                duration: 1000,
                easing: 'easeOutQuart'
            },
            onHover: function(event, elements) {
                const chartOverlay = document.getElementById('chartOverlay');
                if (event.x && event.y) {
                    chartOverlay.style.setProperty('--mouse-x', `${event.x}px`);
                    chartOverlay.style.setProperty('--mouse-y', `${event.y}px`);
                    chartOverlay.style.opacity = elements.length > 0 ? '1' : '0';
                }
                
                // Mettre en surbrillance les barres survolées
                if (elements.length > 0) {
                    const element = elements[0];
                    const meta = mainChart.getDatasetMeta(element.datasetIndex);
                    const bar = meta.data[element.index];
                    
                    // Animation de pulse sur la barre
                    bar._model.backgroundColor = element.datasetIndex === 0 ? '#3a8fb8' : 
                                                  element.datasetIndex === 1 ? '#4a50c4' : '#8a5cd4';
                    mainChart.update('none');
                }
            }
        }
    });
}

function externalTooltip(context) {
    const tooltip = document.getElementById('dataTooltip');
    
    if (context.tooltip.opacity === 0) {
        tooltip.style.opacity = '0';
        return;
    }
    
    const dataIndex = context.tooltip.dataPoints[0].dataIndex;
    const datasetIndex = context.tooltip.dataPoints[0].datasetIndex;
    const value = context.tooltip.dataPoints[0].parsed.y;
    const label = context.tooltip.title[0];
    
    // Calculer les notions mathématiques
    const allValues = mainChart.data.datasets.map(ds => ds.data[dataIndex]);
    const variance = calculateVariance(allValues);
    const stdDev = Math.sqrt(variance);
    const median = calculateMedian(allValues);
    const parityGap = Math.abs(allValues[0] - allValues[1]) / (allValues[0] + allValues[1]) * 100;
    
    const tooltipContent = `
        <div class="tooltip-header">
            <strong>${label}</strong>
            <span class="tooltip-value">${value.toFixed(1)}%</span>
        </div>
        <div class="tooltip-stats">
            <div><span>Variance:</span> ${variance.toFixed(2)}</div>
            <div><span>Écart-type:</span> ${stdDev.toFixed(2)}</div>
            <div><span>Médiane:</span> ${median.toFixed(2)}</div>
            <div><span>Écart parité:</span> ${parityGap.toFixed(1)}%</div>
        </div>
    `;
    
    tooltip.innerHTML = tooltipContent;
    tooltip.style.left = context.tooltip.caretX + 'px';
    tooltip.style.top = context.tooltip.caretY + 'px';
    tooltip.style.opacity = '1';
    tooltip.style.transform = `translate(-50%, calc(-100% - 10px))`;
}

function calculateVariance(values) {
    const mean = values.reduce((a, b) => a + b, 0) / values.length;
    return values.reduce((a, b) => a + Math.pow(b - mean, 2), 0) / values.length;
}

function calculateMedian(values) {
    const sorted = [...values].sort((a, b) => a - b);
    const mid = Math.floor(sorted.length / 2);
    return sorted.length % 2 !== 0 ? sorted[mid] : (sorted[mid - 1] + sorted[mid]) / 2;
}

function loadData() {
    const year = years[currentYearIndex];
    
    // Mettre à jour l'affichage de l'année
    document.querySelector('.highlight-year').childNodes[0].textContent = year + " ";
    
    // Générer des données aléatoires basées sur l'année et le type de graphique
    const baseValue = 50 + (parseInt(year) - 2019) * 5;
    
    mainChart.data.datasets.forEach((dataset, index) => {
        dataset.data = chartConfigs[currentChartIndex].labels.map((_, i) => {
            const randomFactor = 0.8 + Math.random() * 0.4;
            const trendFactor = 1 + (index * 0.1);
            return Math.round((baseValue * randomFactor * trendFactor * (i + 1)) / 2);
        });
    });
    
    // Mettre à jour les indicateurs mathématiques
    updateMathIndicators();
    
    // Animer la mise à jour du graphique
    mainChart.update();
    
    // Appliquer l'animation aux barres
    setTimeout(() => {
        animateChartBars();
    }, 100);
}

function animateChartBars() {
    const meta = mainChart.getDatasetMeta(0);
    if (meta && meta.data) {
        meta.data.forEach((bar, index) => {
            bar._model.backgroundColor = chartConfigs[currentChartIndex].datasets[0].backgroundColor;
            setTimeout(() => {
                bar.transition(1000).easing('easeOutElastic');
            }, index * 50);
        });
    }
}

function updateMathIndicators() {
    const allData = mainChart.data.datasets.flatMap(ds => ds.data);
    const variance = calculateVariance(allData);
    const stdDev = Math.sqrt(variance);
    const median = calculateMedian(allData);
    
    // Mettre à jour les cartes de métriques
    document.querySelector('[data-stat="variance"] .metric-value').textContent = variance.toFixed(2);
    document.querySelector('[data-stat="ecart-type"] .metric-value').textContent = stdDev.toFixed(2);
    document.querySelector('[data-stat="mediane"] .metric-value').textContent = median.toFixed(2);
    
    // Calculer la corrélation (simulée)
    const correlation = 0.5 + Math.random() * 0.3;
    document.querySelector('[data-stat="correlation"] .metric-value').textContent = correlation.toFixed(2);
    
    // Calculer l'écart à la parité
    const maleData = mainChart.data.datasets[0]?.data || [];
    const femaleData = mainChart.data.datasets[1]?.data || [];
    const avgParityGap = maleData.reduce((sum, val, i) => {
        return sum + (Math.abs(val - (femaleData[i] || 0)) / (val + (femaleData[i] || 1)) * 100);
    }, 0) / maleData.length;
    
    document.querySelector('[data-stat="parite"] .metric-value').textContent = avgParityGap.toFixed(1) + '%';
}

function setupEventListeners() {
    // Navigation horizontale des graphiques
    document.getElementById('chart-left').addEventListener('click', () => navigateChart(-1));
    document.getElementById('chart-right').addEventListener('click', () => navigateChart(1));
    
    // Navigation verticale des années
    document.getElementById('year-up').addEventListener('click', () => navigateYear(-1));
    document.getElementById('year-down').addEventListener('click', () => navigateYear(1));
    
    // Changement d'année via le texte
    document.querySelector('.highlight-year').addEventListener('click', () => {
        const yearFilter = document.getElementById('yearFilter');
        yearFilter.style.display = 'block';
        yearFilter.focus();
        yearFilter.click();
        setTimeout(() => {
            yearFilter.style.display = 'none';
        }, 100);
    });
    
    document.getElementById('yearFilter').addEventListener('change', function() {
        const yearIndex = years.indexOf(this.value);
        if (yearIndex !== -1) {
            currentYearIndex = yearIndex;
            loadData();
        }
    });
    
    // Switch mode lab
    document.getElementById('lab-switch').addEventListener('change', function() {
        isLabMode = this.checked;
        document.body.classList.toggle('lab-mode', isLabMode);
        
        // Mettre à jour le texte du bouton
        const labText = document.querySelector('.lab-text');
        labText.textContent = isLabMode ? 'Mode Artistique' : 'Mode Scientifique';
        
        // Animation de transition
        document.querySelector('.toggle-lab label').style.transform = 
            isLabMode ? 'rotate(2deg)' : 'rotate(-2deg)';
        
        setTimeout(() => {
            document.querySelector('.toggle-lab label').style.transform = 'rotate(0deg)';
        }, 200);
        
        // Recharger les données avec le nouveau style
        mainChart.update();
    });
    
    // Interactions au survol des cartes de métriques
    document.querySelectorAll('.metric-card').forEach(card => {
        card.addEventListener('mouseenter', function() {
            this.style.transform = 'translateY(-8px) scale(1.02)';
            this.style.zIndex = '10';
        });
        
        card.addEventListener('mouseleave', function() {
            this.style.transform = 'translateY(0) scale(1)';
            this.style.zIndex = '1';
        });
        
        card.addEventListener('click', function() {
            const stat = this.dataset.stat;
            showStatDetails(stat);
        });
    });
    
    // Navigation au clavier
    document.addEventListener('keydown', (e) => {
        switch(e.key) {
            case 'ArrowLeft':
                navigateChart(-1);
                break;
            case 'ArrowRight':
                navigateChart(1);
                break;
            case 'ArrowUp':
                navigateYear(-1);
                break;
            case 'ArrowDown':
                navigateYear(1);
                break;
            case 'l':
            case 'L':
                document.getElementById('lab-switch').click();
                break;
        }
    });
}

function navigateChart(direction) {
    currentChartIndex = (currentChartIndex + direction + chartConfigs.length) % chartConfigs.length;
    
    // Animation de transition
    const chartArea = document.querySelector('.chart-area');
    chartArea.style.transform = `translateX(${direction * 20}px)`;
    chartArea.style.opacity = '0.7';
    
    setTimeout(() => {
        // Changer le type de graphique
        mainChart.config.type = chartConfigs[currentChartIndex].type;
        mainChart.data.labels = chartConfigs[currentChartIndex].labels;
        
        // Mettre à jour les datasets
        chartConfigs[currentChartIndex].datasets.forEach((config, i) => {
            if (mainChart.data.datasets[i]) {
                Object.assign(mainChart.data.datasets[i], config);
            }
        });
        
        // Recharger les données
        loadData();
        
        // Animation de retour
        chartArea.style.transform = 'translateX(0)';
        chartArea.style.opacity = '1';
    }, 300);
}

function navigateYear(direction) {
    const newIndex = currentYearIndex + direction;
    if (newIndex >= 0 && newIndex < years.length) {
        currentYearIndex = newIndex;
        
        // Animation de transition
        const yearElement = document.querySelector('.highlight-year');
        yearElement.style.transform = `translateY(${direction * 10}px)`;
        yearElement.style.opacity = '0.5';
        
        setTimeout(() => {
            loadData();
            yearElement.style.transform = 'translateY(0)';
            yearElement.style.opacity = '1';
        }, 200);
    }
}

function showStatDetails(stat) {
    const tooltip = document.getElementById('dataTooltip');
    const statTitles = {
        'variance': 'Analyse de Variance',
        'ecart-type': 'Écart-type des Données',
        'correlation': 'Corrélation TV/Radio',
        'mediane': 'Calcul de la Médiane',
        'parite': 'Écart à la Parité'
    };
    
    const statDetails = {
        'variance': 'Mesure la dispersion des données autour de la moyenne. Une variance élevée indique une grande variabilité.',
        'ecart-type': 'Racine carrée de la variance. Indique l\'écart moyen des valeurs par rapport à la moyenne.',
        'correlation': 'Mesure la relation linéaire entre deux variables (TV et Radio). Valeur entre -1 et 1.',
        'mediane': 'Valeur centrale qui sépare la moitié supérieure de la moitié inférieure des données.',
        'parite': 'Différence entre la représentation masculine et féminine par rapport à l\'objectif de parité (50/50).'
    };
    
    tooltip.innerHTML = `
        <div class="tooltip-header">
            <strong>${statTitles[stat]}</strong>
        </div>
        <div class="tooltip-content">
            <p>${statDetails[stat]}</p>
            <div class="tooltip-formula">
                ${getStatFormula(stat)}
            </div>
        </div>
    `;
    
    tooltip.style.opacity = '1';
    tooltip.style.transform = 'translate(-50%, 20px)';
    
    setTimeout(() => {
        tooltip.style.opacity = '0';
    }, 3000);
}

function getStatFormula(stat) {
    const formulas = {
        'variance': 'σ² = Σ(xᵢ - μ)² / N',
        'ecart-type': 'σ = √σ²',
        'correlation': 'r = Σ[(xᵢ - x̄)(yᵢ - ȳ)] / √[Σ(xᵢ - x̄)² Σ(yᵢ - ȳ)²]',
        'mediane': 'Médiane = valeur au rang (N+1)/2',
        'parite': 'Écart = |M - F| / (M + F) × 100%'
    };
    return formulas[stat] || '';
}
